# ---------- bootstrap ----------
#' Bootstrap mixture parameters
#'
#' @param fit fitted object from fit_lognorm2 or fit_norm2
#' @param x numeric vector (if fit not provided)
#' @param par numeric vector of parameters (if fit not provided)
#' @param family "lognormal" or "normal" (if fit not provided)
#' @param B number of bootstrap replicates
#' @param parametric logical, parametric bootstrap if TRUE
#' @param boot_size size or fraction (if between 0 and 1) of bootstrap sample
#' @param parallelType integer for DEoptim/pbapply parallelism
#' @param quiet 0/1/2 for verbosity
#' @param ci_level confidence level
#' @return list with cleaned bootstrap estimates, central tendency, and CI
#' @export
#' @importFrom pbapply pbsapply
bootstrap_mix2 <- function(fit = NULL, x = NULL, par = NULL, family = NULL,
                           B = 1000, parametric = TRUE, boot_size = NULL,
                           parallelType = 0, quiet = 2, ci_level = 0.95) {
  if (!requireNamespace("pbapply", quietly = TRUE)) stop("Install 'pbapply' first")

  if (!is.null(fit)) { x <- fit$data; par <- fit$par; family <- fit$family }
  if (is.null(x) || is.null(par) || is.null(family)) stop("Provide fit or (x, par, family)")
  n <- length(x); if (is.null(boot_size)) boot_size <- n
  if (boot_size > 0 && boot_size < 1) boot_size <- floor(boot_size * n)

  bounds <- if (family == "lognormal") default_bounds_lognorm2(x) else default_bounds_norm2(x)
  lower <- bounds$lower; upper <- bounds$upper
  loglik_fn <- if (family == "lognormal") loglik_lognorm else loglik_norm

  FUN_boot <- function(b) {
    if (parametric) {
      z_b <- rbinom(boot_size, 1, par[1])
      if (family == "lognormal") {
        xb <- ifelse(z_b == 1, rlnorm(boot_size, par[2], par[3]), rlnorm(boot_size, par[4], par[5]))
      } else {
        xb <- ifelse(z_b == 1, rnorm(boot_size, par[2], par[3]), rnorm(boot_size, par[4], par[5]))
      }
    } else xb <- sample(x, boot_size, TRUE)

    tmp_opt <- tryCatch({
      optim(par, fn = function(p) -loglik_fn(p, xb), method = "L-BFGS-B",
            lower = lower, upper = upper,
            control = list(factr = 1e7, pgtol = 1e-8, maxit = 100000))
    }, error = function(e) e)
    if (inherits(tmp_opt, "error")) return(rep(NA_real_, 5))
    else return(order_components(tmp_opt$par))
  }

  # Use pbapply for progress and optional parallelism
  cl <- if (parallelType == 0) 1 else parallelType
  ests_list <- pbapply::pbsapply(seq_len(B), FUN_boot, cl = cl)
  # pbsapply returns a matrix with 5 x B if all succeed, else may vary
  ests <- t(ests_list)
  if (!is.matrix(ests)) ests <- matrix(ests, ncol = 5, byrow = TRUE)
  colnames(ests) <- c("p", "m1", "s1", "m2", "s2")

  # Remove rows with NA
  ests_clean <- ests[apply(ests, 1, function(r) all(!is.na(r))), , drop = FALSE]
  alpha <- 1 - ci_level
  ci <- apply(ests_clean, 2, function(cc) quantile(cc, probs = c(alpha / 2, 1 - alpha / 2), na.rm = TRUE))
  central <- if (family == "lognormal") apply(ests_clean, 2, median, na.rm = TRUE) else apply(ests_clean, 2, mean, na.rm = TRUE)

  list(ests_clean = ests_clean, central = central, ci = ci, family = family)
}
