% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ml_helper.R
\name{ml_backtest_multi}
\alias{ml_backtest_multi}
\title{Run multi-horizon ML backtests (pooled or sector-neutral)}
\usage{
ml_backtest_multi(
  features_list,
  prices_weekly,
  horizons,
  fit_fn,
  predict_fn,
  schedule,
  transform = "zscore",
  selection = list(top_k = 20L),
  weighting = list(method = "softmax", temperature = 12),
  caps = list(max_per_symbol = 0.1),
  group_mode = c("pooled", "per_group"),
  group_map = NULL,
  initial_capital = 1e+05,
  name_prefix = "",
  seed = NULL,
  ...
)
}
\arguments{
\item{features_list}{Named list of data.tables with factor scores (each with a
\code{Date} column and one column per symbol). Typically from
\code{\link[=ml_prepare_features]{ml_prepare_features()}}.}

\item{prices_weekly}{Wide price table (weekly) with \code{Date} + one column per
symbol (adjusted prices). Used both to create labels and run the backtest.}

\item{horizons}{Integer vector of horizons in \emph{weeks} (e.g., \code{c(1L,4L,13L)}).}

\item{fit_fn, predict_fn}{Model fit/predict closures as returned by
\code{\link[=ml_make_model]{ml_make_model()}}.}

\item{schedule}{Walk-forward schedule list with elements \code{is}, \code{oos}, \code{step}.}

\item{transform}{Feature transform (default \code{"zscore"}). Passed to
\code{\link[=ml_backtest]{ml_backtest()}}.}

\item{selection}{List describing selection rules (e.g., \code{list(top_k=20L, max_per_group=3L)}).}

\item{weighting}{List describing weighting rules (e.g., \code{list(method="softmax", temperature=12)}).}

\item{caps}{List with exposure caps (e.g., \code{list(max_per_symbol=0.10, max_per_group=NULL)}).}

\item{group_mode}{\code{"pooled"} or \code{"per_group"}. If \code{"per_group"}, you must pass \code{group_map}.}

\item{group_map}{A two-column table with columns \code{Symbol} and \code{Group} defining
the grouping (e.g., sectors) for \code{"per_group"} mode.}

\item{initial_capital}{Numeric. Starting capital for the backtest (default \code{1e5}).}

\item{name_prefix}{Optional string prefixed to each backtest title.}

\item{seed}{Optional integer. If provided, the same seed is set before each
horizon’s backtest call to ensure deterministic tie-breaks.}

\item{...}{Additional arguments forwarded to \code{\link[=ml_backtest]{ml_backtest()}}
(kept for future extensibility; no effect if unused).}
}
\value{
A named list of backtest objects (as returned by
\code{\link[=ml_backtest]{ml_backtest()}}), with names like \code{"H1w"}, \code{"H4w"}, … .
}
\description{
Convenience wrapper around \code{\link[=ml_backtest]{ml_backtest()}} that repeats the
same specification across multiple horizons, returning a named list of
backtest objects keyed as \code{"H1w"}, \code{"H4w"}, \code{"H13w"}, etc.
}
\details{
This function does \strong{not} change core behavior; it only removes boilerplate
when running identical specs across horizons and (optionally) grouping
regimes. It preserves all defaults you pass for selection, weighting,
transforms, caps, and schedule.
}
\examples{
\donttest{
library(PortfolioTesteR)
data(sample_prices_weekly, package = "PortfolioTesteR")

# Minimal features for the example
X <- ml_prepare_features(
  prices_weekly = sample_prices_weekly,
  include = c("mom12","mom26")
)

# Simple deterministic model
model <- ml_make_model("linear")
sched <- list(is = 156L, oos = 4L, step = 4L)

set.seed(42)
bt_list <- ml_backtest_multi(
  features_list = X,
  prices_weekly = sample_prices_weekly,
  horizons = c(1L, 4L),
  fit_fn = model$fit,
  predict_fn = model$predict,
  schedule = sched,
  selection = list(top_k = 5L),
  weighting = list(method = "softmax", temperature = 12),
  caps = list(max_per_symbol = 0.10),
  group_mode = "pooled",
  name_prefix = "Demo ",
  seed = 42
)

names(bt_list)   # "H1w" "H4w"
}

}
