% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/process-plate.R
\name{process_plate}
\alias{process_plate}
\title{Process Plate Data and Save Normalised Output}
\usage{
process_plate(
  plate,
  filename = NULL,
  output_dir = "normalised_data",
  write_output = TRUE,
  normalisation_type = "RAU",
  data_type = "Median",
  sample_type_filter = "ALL",
  blank_adjustment = FALSE,
  verbose = TRUE,
  reference_dilution = 1/400,
  ...
)
}
\arguments{
\item{plate}{A \link{Plate} object containing raw or processed Luminex data.}

\item{filename}{(\code{character(1)}, optional)
Output CSV filename. If \code{NULL}, defaults to \code{"{plate_name}_{normalisation_type}.csv"}.
File extension is auto-corrected to \code{.csv} if missing. If an absolute path is given,
\code{output_dir} is ignored.}

\item{output_dir}{(\code{character(1)}, default = \code{"normalised_data"})
Directory where the CSV will be saved. Will be created if it doesn't exist.
If \code{NULL}, the current working directory is used.}

\item{write_output}{(\code{logical(1)}, default = \code{TRUE})
Whether to write the output to disk.}

\item{normalisation_type}{(\code{character(1)}, default = \code{'RAU'}) The normalisation method to apply.
\itemize{
\item Allowed values: \code{c(MFI, RAU, nMFI)}.
}}

\item{data_type}{(\code{character(1)}, default = \code{"Median"})
The data type to use for normalisation (e.g., \code{"Median"}).}

\item{sample_type_filter}{(\code{character()}) The types of samples to normalise.
(e.g., \code{"TEST"}, \code{"STANDARD CURVE"}). It can also be a vector of sample types.
In that case, dataframe with multiple sample types will be returned.
By default equals to \code{"ALL"}, which corresponds to processing all sample types.}

\item{blank_adjustment}{(\code{logical(1)}, default =  \code{FALSE}) Whether to apply blank adjustment before processing.}

\item{verbose}{(\code{logical(1)}, default = \code{TRUE}) Whether to print additional information during execution.}

\item{reference_dilution}{(\code{numeric(1)} or \code{character(1)}, default = \code{1/400})
Target dilution used for nMFI calculation. Ignored for other types.
Can be numeric (e.g., \code{0.0025}) or string (e.g., \code{"1/400"}).}

\item{...}{Additional arguments passed to the model fitting function \code{\link[=create_standard_curve_model_analyte]{create_standard_curve_model_analyte()}} and \code{\link{predict.Model}}}
}
\value{
A data frame of computed values, with test samples as rows and analytes as columns.
}
\description{
Processes a Luminex plate and computes normalised values using the specified
\code{normalisation_type}. Depending on the chosen method, the function performs
blank adjustment, fits models, and extracts values for test samples.
Optionally, the results can be saved as a CSV file.
}
\details{
Supported normalisation types:
\itemize{
\item \strong{RAU} (Relative Antibody Units):
Requires model fitting. Produces estimates using a standard curve.
See \code{\link{create_standard_curve_model_analyte}} for details.
\item \strong{nMFI} (Normalised Median Fluorescence Intensity):
Requires a reference dilution. See \code{\link{get_nmfi}}.
\item \strong{MFI} (Blank-adjusted Median Fluorescence Intensity):
Returns raw MFI values (adjusted for blanks, if requested).
}
}
\section{RAU Workflow}{

\enumerate{
\item Optionally perform blank adjustment.
\item Fit a model for each analyte using standard curve data.
\item Predict RAU values for test samples.
\item Aggregate and optionally save results.
}
}

\section{nMFI Workflow}{

\enumerate{
\item Optionally perform blank adjustment.
\item Compute normalised MFI using the \code{reference_dilution}.
\item Aggregate and optionally save results.
}
}

\section{MFI Workflow}{

\enumerate{
\item Optionally perform blank adjustment.
\item Return adjusted MFI values.
}
}

\examples{
plate_file <- system.file("extdata", "CovidOISExPONTENT_CO_reduced.csv", package = "SerolyzeR")
layout_file <- system.file("extdata", "CovidOISExPONTENT_CO_layout.xlsx", package = "SerolyzeR")
plate <- read_luminex_data(plate_file, layout_file, verbose = FALSE)

example_dir <- tempdir(check = TRUE)

# Process using default settings (RAU normalisation)
process_plate(plate, output_dir = example_dir)

# Use a custom filename and skip blank adjustment
process_plate(plate,
  filename = "no_blank.csv",
  output_dir = example_dir,
  blank_adjustment = FALSE
)

# Use nMFI normalisation with reference dilution
process_plate(plate,
  normalisation_type = "nMFI",
  reference_dilution = "1/400",
  output_dir = example_dir
)

}
\seealso{
\code{\link{create_standard_curve_model_analyte}}, \code{\link{get_nmfi}}
}
