% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/descriptive.plots.R
\name{descriptive.plots}
\alias{descriptive.plots}
\title{Descriptive plots for the assessment of agreement}
\usage{
descriptive.plots(data, y, id, met, rep = NA,
                  plots = c("scatterplot", "boxplot",
                            "histogram.y", "qqplot.y",
                            "histogram.d", "qqplot.d"),
                  rater.col = c("#cc6e5c", "#06402b"), d.col = "lavender",
                  scatterplot.xlab = NULL, scatterplot.ylab = NULL,
                  histogram.y.main = NULL, histogram.y.xlab = NULL,
                  qqplot.y.main = NULL,
                  line.col = "red",
                  histogram.d.density = TRUE, histogram.d.bw = "nrd0",
                  density.col = "red", ...)
}
\arguments{
\item{data}{name of the dataset, of class \code{data.frame}, containing at least 3 columns (quantitative measurement, subject effect, rater effect).}

\item{y}{quantitative measurement column name.}

\item{id}{subject effect column name. The corresponding column of \code{data} must be a factor.}

\item{met}{rater effect column name. The corresponding column of \code{data} must be a factor.}

\item{rep}{replicate effect column name. When there are no replicates the user should use \code{rep = NA}. When there are replicates, the corresponding column of data must be a factor. \cr
The default value is \code{NA}.}

\item{plots}{name of the descriptive plots to display. The options are:
\code{"scatterplot"} (a scatter plot of all the possible paired-measurements of one rater against the other), \code{"boxplot"} (a boxplot of the measurements per rater and, if it applies, replicate),
\code{"histogram.y"} (a histogram of the measurements per rater and, if it applies, replicate), \code{"qqplot.y"} (a QQ plot of the measurements per rater and, if it applies, replicate),
\code{"histogram.d"} (a histogram of the paired-differences) and \code{"qqplot.d"} (a QQ plot of the paired-differences).
This argument is not case-sensitive and is passed to \code{\link[base]{match.arg}}. \cr
The default value is \code{c("scatterplot", "boxplot", "histogram.y", "qqplot.y", "histogram.d", "qqplot.d")}, so all plots are executed by default.}

\item{rater.col}{colour palette to be used in the plots involving the measurements separated per rater. If neither \code{"boxplot"} nor \code{"histogram.y"} are selected in \code{plots}, this argument is ignored. \cr
The default value is \code{c("#cc6e5c", "#06402b")} for the first and second level, respectively, of the variable met in \code{data}.}

\item{d.col}{colour to be used in the histogram involving the paired-differences. If \code{"histogram.d"} is not selected in \code{plots}, this argument is ignored. \cr
The default value is \code{"lavender"}.}

\item{scatterplot.xlab}{a label to use for the x-axis in the scatterplot. If \code{"scatterplot"} is not selected in \code{plots}, this argument is ignored. \cr
The default value, \code{NULL}, indicates that the label \emph{"}\code{y}\emph{" from "1st level of} \code{met} \emph{column"} should be used.}

\item{scatterplot.ylab}{a label to use for the y-axis in the scatterplot. If \code{"scatterplot"} is not selected in \code{plots}, this argument is ignored. \cr
The default value, \code{NULL}, indicates that the label \emph{"}\code{y}\emph{" from "2nd level of} \code{met} \emph{column"} should be used.}

\item{histogram.y.main}{overall title for the histograms of the measurements (to be passed to \code{main} argument in \code{\link[graphics]{hist}}).
Notice that all histograms (one per rater and, if it applies, replicate) will have the same title.
If \code{"histogram.y"} is not selected in \code{plots}, this argument is ignored. \cr
The default value, \code{NULL}, indicates that the label \emph{"\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of no replicates, where \eqn{j \in} \{1,2\}, and the label
\emph{"\eqn{i}-th level of} \code{rep} \emph{column" from "\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of replicates, where \eqn{i \in} \{1,..., number of replicates\} and \eqn{j \in} \{1,2\}.}

\item{histogram.y.xlab}{a label to use for the x-axis in the histograms of the measurements (to be passed to \code{xlab} argument in \code{\link[graphics]{hist}}).
Notice that all histograms (one per rater and, if it applies, replicate) will have the same label.
If \code{"histogram.y"} is not selected in \code{plots}, this argument is ignored. \cr
The default value, \code{NULL}, indicates that the label \emph{"}\code{y}\emph{"} should be used.}

\item{qqplot.y.main}{overall title for the normal quantile-quantile plots of the measurements (to be passed to \code{main} argument in \code{\link[stats]{qqplot}}).
Notice that all QQ plots (one per rater and, if it applies, replicate) will have the same title.
If \code{"qqplot.y"} is not selected in \code{plots}, this argument is ignored. \cr
The default value, \code{NULL}, indicates that the label \emph{"\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of no replicates, where \eqn{j \in} \{1,2\}, and the label
\emph{"\eqn{i}-th level of} \code{rep} \emph{column" from "\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of replicates, where \eqn{i \in} \{1,..., number of replicates\} and \eqn{j \in} \{1,2\}.}

\item{line.col}{colour to be used in the theoretical line added to the scatterplot or the normal quantile-quantile plots. If none of \code{"scatterplot"}, \code{"qqplot.y"}, or \code{"qqplot.d"} are selected in \code{plots}, this argument is ignored. \cr
The default value is \code{"red"}.}

\item{histogram.d.density}{logical indicating whether the density should be added as a solid curve to the histogram of the paired-differences.
If \code{"histogram.d"} is not selected in \code{plots}, this argument is ignored. \cr
The default value is \code{TRUE}.}

\item{histogram.d.bw}{the smoothing bandwidth to be used in the density curve added to the histogram involving the paired differences (to be passed to \code{bw} argument in \code{\link[stats]{density}}).
If \code{"histogram.d"} is not selected in \code{plots} or \code{histogram.d.density} is \code{FALSE}, this argument is ignored. \cr
The default value is \code{"nrd0"}.}

\item{density.col}{colour to be used in the density curve added to the histogram involving the paired differences. If \code{"histogram.d"} is not selected in \code{plots} or \code{histogram.d.density} is \code{FALSE}, this argument is ignored. \cr
The default value is \code{"red"}.}

\item{...}{other graphical parameters (to be passed to each plot). To be used calling only one plot for optimal performance.}
}
\value{
The desired plots are shown one after the other.
}
\description{
This function implements some basic descriptive plots that can be analysed in the context of agreement. The plots implemented are:
\itemize{
   \item{a scatter plot of all the possible paired-measurements of one rater against the other}
   \item{a boxplot of the measurements per rater and, if it applies, replicate}
   \item{a histogram of the measurements per rater and, if it applies, replicate}
   \item{a normal quantile-quantile plot of the measurements per rater and, if it applies, replicate}
   \item{a histogram of the paired-differences}
   \item{a normal quantile-quantile plot of the paired-differences}}
}
\examples{
# normal data

set.seed(2025)

n <- 100

mu.ind <- rnorm(n, 0, 7)

epsA1 <- rnorm(n, 0, 3)
epsA2 <- rnorm(n, 0, 3)
epsB1 <- rnorm(n, 0, 3)
epsB2 <- rnorm(n, 0, 3)

y_A1 <- 50 + mu.ind + epsA1 # rater A, replicate 1
y_A2 <- 50 + mu.ind + epsA2 # rater A, replicate 2
y_B1 <- 40 + mu.ind + epsB1 # rater B, replicate 1
y_B2 <- 40 + mu.ind + epsB2 # rater B, replicate 2

ex_data <- data.frame(y = c(y_A1, y_A2, y_B1, y_B2),
                      rater = factor(rep(c("A", "B"), each = 2*n)),
                      replicate = factor(rep(rep(1:2, each = n), 2)),
                      subj = factor(rep(1:n, 4)))

descriptive.plots(ex_data, y, subj, rater, replicate, plots = "scatterplot")
descriptive.plots(ex_data, y, subj, rater, replicate, plots = "boxplot",
                  rater.col = c("blue", "red"), names = rep(c("1st rep.", "2nd rep."), 2),
                  xlab = "", main = "Boxplots per rater and replicate")
legend("topright", legend = c("A", "B"), fill = c("blue", "red"))

# non-normal data

# involving the measurements
def.par <- par(no.readonly = TRUE)
par(mfcol = c(4, 2), las = 1)
descriptive.plots(AMLad, mrd, id, met, rep, plots = "histogram.y",
                  ylim = c(0, 120), xlim = c(0, 100),
                  breaks = seq(from = 0, to = 100, by = 2.5))
descriptive.plots(AMLad, mrd, id, met, rep, plots = "qqplot.y",
                  ylim = c(0, 8), xlim = c(-3, 3))
par(def.par)

# inolving the paired-differences
def.par <- par(no.readonly = TRUE)
par(mfrow = c(1, 2), las = 1)
descriptive.plots(AMLad, mrd, id, met, rep, plots = "histogram.d",
                  histogram.d.bw = 2,
                  ylim = c(0, 0.15), breaks = seq(-20, 100, 5),
                  xlab = "Paired-differences", main = "",
                  line.col = "darkred", density.col = "darkred")
descriptive.plots(AMLad, mrd, id, met, rep, plots = "qqplot.d",
                  ylim = c(-2, 8), xlim = c(-4, 4), main = "")
par(def.par)

}
