% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/class_constr.R
\name{simdesign}
\alias{simdesign}
\title{Construct a new simdesign object}
\usage{
simdesign(
  simmethod = character(),
  siminput = tibble::tibble(),
  simobject = list(),
  simseeds = NA_integer_,
  simoutput = tibble::tibble(),
  ...
)
}
\arguments{
\item{simmethod}{character string defining the method of the simulation design}

\item{siminput}{tibble providing input parameterisations for the NetLogo model (cols=parameter, rows=runs)}

\item{simobject}{used for some methods to store additional information (sobol, morris, eFast)}

\item{simseeds}{a vector or model random seeds}

\item{simoutput}{tibble containing model results}

\item{...}{...}
}
\value{
simdesign S4 class object
}
\description{
Construct a new simdesign object
}
\details{
The simulation design class holds information on the input parameter design of model simulations.
It also stores information that is needed to run method specific analysis functions.
The simseeds can be used to run all model simulations that are defined within the siminput tibble several times with changing random-seeds.
While it is possible to add simdesign directly with this function, we suggest to use our simdesign_helper functions.
A simulation design can be attached to a nl object by using one of these simdesign_helper functions on an already defined \link[nlrx]{nl}
object with a valid \link[nlrx]{experiment}.
All simdesign helpers use the defined constants and variables of the experiment to create the siminput tibble.
NetLogo parameters that are not defined in constants or variables will be set with their default value from the NetLogo interface.

Currently, following simdesign_helper functions are provided:

\link[nlrx]{simdesign_simple}

The simple simdesign only uses defined constants and reports a parameter matrix with only one parameterization.
To setup a simple simdesign, no variables have to be defined.

\link[nlrx]{simdesign_distinct}

The distinct simdesign can be used to run distinct parameter combinations.
To setup a distinct simdesign, vectors of values need to be defined for each variable.
These vectors must have the same number of elements across all variables.
The first simulation run consist of all 1st elements of these variable vectors; the second run uses all 2nd values, and so on.

\link[nlrx]{simdesign_ff}

The full factorial simdesign creates a full-factorial parameter matrix with all possible combinations of parameter values.
To setup a full-factorial simdesign, vectors of values need to be defined for each variable.
Alternatively, a sequence can be defined by setting min, max and step.
However, if both (values and min, max, step) are defined, the values vector is prioritized.

\link[nlrx]{simdesign_lhs}

The latin hypercube simdesign creates a Latin Hypercube sampling parameter matrix.
The method can be used to generate a near-random sample of parameter values from the defined parameter distributions.
More Details on Latin Hypercube Sampling can be found in McKay (1979) \doi{10.1080/00401706.1979.10489755}.
nlrx uses the \href{https://CRAN.R-project.org/package=lhs/index.html}{lhs} package to generate the Latin Hypercube parameter matrix.
To setup a latin hypercube sampling simdesign, variable distributions need to be defined (min, max, qfun).

Sensitivity Analyses: \link[nlrx]{simdesign_sobol}, \link[nlrx]{simdesign_sobol2007}, \link[nlrx]{simdesign_soboljansen}, \link[nlrx]{simdesign_morris}, \link[nlrx]{simdesign_eFast}

Sensitivity analyses are useful to estimate the importance of model parameters and to scan the parameter space in an efficient way.
nlrx uses the \href{https://CRAN.R-project.org/package=sensitivity/index.html}{sensitivity} package to setup sensitivity analysis parameter matrices.
All supported sensitivity analysis simdesigns can be used to calculate sensitivity indices for each parameter-output combination.
These indices can be calculated by using the \link[nlrx]{analyze_nl} function after attaching the simulation results to the nl object.
To setup sensitivity analysis simdesigns, variable distributions (min, max, qfun) need to be defined.

Optimization techniques: \link[nlrx]{simdesign_GenSA}, \link[nlrx]{simdesign_GenAlg}

Optimization techniques are a powerful tool to search the parameter space for specific solutions.
Both approaches try to minimize a specified model output reporter by systematically (genetic algorithm, utilizing the \href{https://CRAN.R-project.org/package=genalg/index.html}{genalg} package) or randomly (simulated annealing, utilizing the \href{https://CRAN.R-project.org/package=GenSA/index.html}{genSA} package) changing the model parameters within the allowed ranges.
To setup optimization simdesigns, variable ranges (min, max) need to be defined.
Optimization simdesigns can only be executed using the \link[nlrx]{run_nl_dyn} function instead of \link[nlrx]{run_nl_all} or \link[nlrx]{run_nl_one}.
}
\examples{

# To attach a simdesign, a nl object needs to be created first (see ?nl).
# For this example, we load nl objects from test data.

# Simdesign examples for Wolf Sheep Predation model from NetLogo models library:

nl <- nl_simple
nl@simdesign <- simdesign_simple(nl = nl,
                                 nseeds = 3)

nl <- nl_distinct
nl@simdesign <- simdesign_distinct(nl = nl,
                                   nseeds = 3)

nl <- nl_ff
nl@simdesign <- simdesign_ff(nl = nl,
                             nseeds = 3)

nl <- nl_lhs
nl@simdesign <- simdesign_lhs(nl=nl,
                              samples=100,
                              nseeds=3,
                              precision=3)

nl <- nl_sobol
nl@simdesign <- simdesign_sobol(nl=nl,
                                samples=200,
                                sobolorder=2,
                                sobolnboot=20,
                                sobolconf=0.95,
                                nseeds=3,
                                precision=3)

nl <- nl_sobol2007
nl@simdesign <- simdesign_sobol2007(nl=nl,
                                    samples=200,
                                    sobolnboot=20,
                                    sobolconf=0.95,
                                    nseeds=3,
                                    precision=3)

nl <- nl_soboljansen
nl@simdesign <- simdesign_soboljansen(nl=nl,
                                      samples=200,
                                      sobolnboot=20,
                                      sobolconf=0.95,
                                      nseeds=3,
                                      precision=3)

nl <- nl_morris
nl@simdesign <- simdesign_morris(nl=nl,
                                 morristype="oat",
                                 morrislevels=4,
                                 morrisr=100,
                                 morrisgridjump=2,
                                 nseeds=3)

nl <- nl_eFast
nl@simdesign <- simdesign_eFast(nl=nl,
                                samples=100,
                                nseeds=3)

nl <- nl_lhs
nl@simdesign <- simdesign_GenAlg(nl=nl,
                                 popSize = 200,
                                 iters = 100,
                                 evalcrit = 1,
                                 elitism = NA,
                                 mutationChance = NA,
                                 nseeds = 1)

nl <- nl_lhs
nl@simdesign <- simdesign_GenSA(nl=nl,
                                par=NULL,
                                evalcrit=1,
                                control=list(max.time = 600),
                                nseeds=1)


}
