% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/np_quantile_causality.R
\name{np_quantile_causality}
\alias{np_quantile_causality}
\alias{lrq_causality_test}
\title{Nonparametric Causality-in-Quantiles Test}
\usage{
np_quantile_causality(x, y, type = c("mean", "variance"), q = NULL, hm = NULL)
}
\arguments{
\item{x}{numeric vector; candidate cause (independent) variable. The test internally uses the
\strong{first lag} of x (one-lag Granger causality setup).}

\item{y}{numeric vector; effect (dependent) variable. The test internally uses the
\strong{first lag} of y (one-lag Granger causality setup).}

\item{type}{character; "mean" or "variance" (causality in mean or variance).}

\item{q}{numeric vector of quantiles in (0,1). Default is seq(0.01, 0.99, 0.01).}

\item{hm}{optional numeric bandwidth; if \code{NULL}, uses Yu & Jones (1998) style
plug-in via \code{KernSmooth::dpill} on the mean-regression proxy.}
}
\value{
An object of class \code{np_quantile_causality} with elements:
\itemize{
\item \code{statistic}: numeric vector of test statistics by quantile
\item \code{quantiles}: numeric vector of quantiles tested
\item \code{bandwidth}: scalar base bandwidth used before quantile adjustment
\item \code{type}: "mean" or "variance"
\item \code{n}: effective sample size
\item \code{call}: the matched call
}
}
\description{
Computes the Balcilar-Jeong-Nishiyama style nonparametric quantile
Granger-causality test for first-order lags. Methodology is based on
Balcilar, Gupta, and Pierdzioch (2016, \doi{10.1016/j.resourpol.2016.04.004})
and Balcilar et al. (2016, \doi{10.1007/s11079-016-9388-x}).
}
\details{
Uses local polynomial quantile regression at each quantile with kernel weights,
constructs the Song et al. (2012) style quadratic form, and rescales to the
asymptotic standard-normal statistic.
}
\note{
This function tests whether \eqn{x_{t-1}} Granger-causes \eqn{y_t}
in quantile \eqn{\theta} (and, with \code{type = "variance"}, whether
\eqn{x_{t-1}^2} causes \eqn{y_t^2}). Higher-order lags are \strong{not} supported
in this release.
}
\section{Lag order (important)}{

The current implementation \strong{uses one lag} of each series only:
\eqn{x_{t-1}} and \eqn{y_{t-1}} (first-order Granger setup).
Extending to higher lags requires changing the internal embedding
(currently \verb{stats::embed(*, 2)}) and the kernel construction to handle
multivariate lag vectors (e.g., a product kernel over all lag coordinates
or a multivariate Gaussian kernel).
}

\section{References}{

\itemize{
\item Balcilar, M., Gupta, R., & Pierdzioch, C. (2016).
Does uncertainty move the gold price? New evidence from a nonparametric
causality-in-quantiles test. \emph{Resources Policy}, 49, 74–80.
\doi{10.1016/j.resourpol.2016.04.004}
\item Balcilar, M., Gupta, R., Kyei, C., & Wohar, M. E. (2016).
Does economic policy uncertainty predict exchange rate returns and volatility?
Evidence from a nonparametric causality-in-quantiles test.
\emph{Open Economies Review}, 27(2), 229–250.
\doi{10.1007/s11079-016-9388-x}
}
}

\examples{
\donttest{
set.seed(1234)
x <- arima.sim(n = 600, list(ar = 0.4))
y <- 0.5*lag(x, -1) + rnorm(600)  # x Granger-causes y
y[is.na(y)] <- mean(y, na.rm = TRUE)
obj <- np_quantile_causality(x, y, type = "mean", q = seq(0.1, 0.9, 0.1))
plot(obj)  # test statistic vs quantiles with 5\% CV line

# Example with bundled dataset (Gold causes Gold or Oil depending on call)
data(gold_oil)
# use first 500 days
gold_oil <- gold_oil[1:501,]
q_grid <- seq(0.25, 0.75, by = 0.25)

# Causality in conditional mean (does Oil_t-1 cause Gold_t?)
res_mean <- np_quantile_causality(
  x = gold_oil$Oil,
  y = gold_oil$Gold,
  type = "mean",
  q = q_grid
)
res_mean

# Causality in conditional variance
res_var <- np_quantile_causality(
  x = gold_oil$Oil,
  y = gold_oil$Gold,
  type = "variance",
  q = q_grid
)
res_var

# Plot (with 5\% critical value line); returns a ggplot object invisibly
plot(res_mean)
plot(res_var)
}

}
