#' @title  Iterative Attribute Hierarchy Exploration Methods for Cognitive Diagnosis Models
#' 
#' @description
#' This function implements an exploratory method for attribute hierarchy structure (Zhang et al., 2025).
#' The procedure is iterative: in each step any structural parameter that is
#' \itemize{
#'   \item less than or equal to the pre-specified threshold \code{eps}, or
#'   \item not greater than 0 (tested by z-statistic under Bonferroni correction and standard erros are from XPD information matrix)
#' }
#' is fixed to zero and the remaining parameters are re-estimated.
#' Upon convergence, all structural parameters are both greater than \code{eps}
#' and significantly larger than 0. The attribute hierarchy is then inferred
#' from the set of parameters that remain significantly positive.
#' 
#' @param Y A required \eqn{N} × \eqn{I} matrix or \code{data.frame} consisting of the responses of 
#'          \code{N} individuals to \eqn{I} items. Missing values need to be coded as \code{NA}.
#' @param Q A required binary \eqn{I} × \eqn{K} matrix containing the attributes not required or required 
#'          master the items. The \code{i}th row of the matrix is a binary indicator vector indicating which
#'          attributes are not required (coded by 0) and which attributes are required (coded by 1) to master
#'          item \eqn{i}.
#' @param model Type of model to be fitted; can be \code{"GDINA"}, \code{"LCDM"}, \code{"DINA"}, \code{"DINO"},
#'              \code{"ACDM"}, \code{"LLM"}, or \code{"rRUM"}. Default = \code{"GDINA"}.
#' @param mono.constraint Logical indicating whether monotonicity constraints should be fulfilled in estimation.
#'                        Default = \code{FALSE}.
#' @param maxitr Number of max iterations. Default = \code{20}.
#' @param eps Cut-off points of the minimum vallue of structural parameters.
#' @param alpha.level alpha level for the z-statistic test under Bonferroni correction. Default = \code{0.01}.
#' @param verbose Logical indicating to print iterative information or not. Default is \code{TRUE}
#' 
#' @return
#' An object of class \code{att.hierarchy} containing the following components:
#' \describe{
#'   \item{\code{statistic}}{A 4-column \code{matrix} for each structural parameter 
#'                           that is significantly larger than 0: the parameter estimate, its 
#'                           standard error (SE), the corresponding z-statistic, and the p-values after 
#'                           Bonferroni correction.}
#'   \item{\code{noSig}}{A \code{logical} scalar: \code{TRUE} if, during iteration, all structural 
#'                       parameters are not greater than 0; otherwise \code{FALSE}.}
#'   \item{\code{isNonverge}}{A \code{logical} scalar: \code{TRUE} if convergence was achieved 
#'               within \code{maxitr} iterations; \code{FALSE} if the algorithm did not converged.}
#'   \item{\code{pattern}}{The attribute mastery pattern matrix that contains every possible attribute 
#'                         mastery pattern.}
#'   \item{\code{arguments}}{A list that stores all input arguments supplied by the user.}
#' }
#' 
#' @author Haijiang Qin <Haijiang133@outlook.com>
#'
#' @references
#' Zhang, X., Jiang, Y., Xin, T., & Liu, Y. (2025). Iterative Attribute Hierarchy Exploration Methods for Cognitive Diagnosis Models. Journal of Educational and Behavioral Statistics, 50(4), 682-713. https://doi.org/10.3102/10769986241268906 
#' 
#' @export
#' @importFrom GDINA attributepattern
#' @importFrom MASS ginv
#' @importFrom stats pnorm p.adjust
#' 
att.hierarchy <- function(Y, Q, model="GDINA", mono.constraint=FALSE, 
                          maxitr=20, eps=1e-7, alpha.level=0.01, verbose=TRUE){
  
  att.hierarchycall <- match.call()
  
  Y <- as.matrix(Y)
  Q <- as.matrix(Q)
  I <- nrow(Q)
  K <- ncol(Q)
  N <- nrow(Y)
  
  pattern.pre <- matrix(0, ncol = ncol(Q))
  pattern.cur <- attributepattern(ncol(Q))
  noSig <- FALSE
  isNonverge <- FALSE
  iter <- 0
  while (!identical(pattern.cur, pattern.pre) & iter < maxitr) {
    iter <- iter + 1
    if(verbose){
      cat(paste0("----------------iteration: ", iter, "----------------\n"))
    }
    
    pattern.pre <- pattern.cur
    CDM.obj <- CDM(Y, Q, model = model, att.str=pattern.pre,
                   method = "EM", maxitr = 2000, verbose = FALSE, mono.constraint = mono.constraint)
    
    P.alpha <- CDM.obj$P.alpha
    
    parameters.mini <- which(P.alpha <= eps)
    if(length(parameters.mini) > 0){
      if(verbose){
        out <- names(parameters.mini)
        names(out) <- 1:length(out)
        cat("some structural parameters lower than eps", paste0("(", eps, "):\n"), out, "\n")
      }
      pattern.pre <- pattern.pre[parameters.mini, ]
      CDM.obj <- CDM(Y, Q, model = model, att.str=pattern.pre,
                     method = "EM", maxitr = 2000, verbose = FALSE, mono.constraint = mono.constraint)
      P.alpha <- CDM.obj$P.alpha
    }
    
    sco <- score(CDM.obj$analysis.obj, parm="prob")
    crossprod.matrix <- crossprod(do.call(cbind, sco))
    crossprod.matrix[which(is.na(crossprod.matrix))] <- 1e-20
    total.se <- sqrt(diag(MASS::ginv(crossprod.matrix)))
    P.alpha.se <- total.se[(length(total.se)-nrow(pattern.pre)+2):length(total.se)]
    
    Z.score <- P.alpha[-1] / P.alpha.se
    
    p_raw <- (1-pnorm(Z.score))
    p_bon <- p.adjust(p_raw, method = "bonferroni")
    parameters.notSig <- which(p_bon >= alpha.level)
    if(length(parameters.notSig) > 0){
      if(all(p_bon*2 >= alpha.level)){
        cat("all structural parameters are not significantly different from 0.\n")
        noSig <- TRUE
        break
      }
      if(verbose){
        out <- names(parameters.notSig)
        names(out) <- 1:length(out)
        cat("some structural parameters are not significantly different from 0:\n", out, "\n")
      }
    }
    
    pattern.cur <- pattern.pre[c(1, which(p_bon < alpha.level)+1), ]
    if(all(rowSums(pattern.cur) != ncol(Q))){
      pattern.cur <- rbind(pattern.cur, rep(1, ncol(Q)))
      if(verbose){
        cat(paste0(rep(1, K), collapse = ""), "is not significantly different from 0:\n", 
            "but it was still included for the next parameter estimation!\n")
      }
    }
    
    if(identical(pattern.cur, pattern.pre)){
      cat("Converged, iteration stopped.\n")
      isNonverge <- TRUE
      break
    }
  }
  
  results <- cbind(P.alpha[-1], P.alpha.se, Z.score, p_bon)
  colnames(results) <- c("est.", "SE", "z", "p.value")
  rownames(results) <- names(Z.score)
  
  att.hierarchy.obj <- list(statistic=results, 
                            pattern=pattern.cur, 
                            noSig=noSig, 
                            isNonverge=isNonverge, 
                            call=att.hierarchycall, 
                            arguments = list(
                              Y=Y, Q=Q, model=model, 
                              mono.constraint=mono.constraint, 
                              maxitr=maxitr, eps=eps, 
                              alpha.level=alpha.level, 
                              verbose=verbose
                            ))
  
  class(att.hierarchy.obj) <- "att.hierarchy"
  return(att.hierarchy.obj)
}
