#' Write data to a table using Databricks Volume
#'
#' This function first writes parquet file to a temporary Databricks Volume and
#' then converts it to a table.
#'
#' @param connector_object A [ConnectorDatabricksTable] object for interacting
#' with Databricks
#' @param x The data to be written to the table
#' @param name The name of the table
#' @param overwrite Overwrite existing content if it exists in the connector.
#' @param tags Named list containing tag names and tag values, e.g.
#' list("tag_name1" = "tag_value1", "tag_name2" = "tag_value2").
#' More info [here](https://docs.databricks.com/aws/en/database-objects/tags)
#' @return None
#' @noRd
#' @examples
#' \dontrun{
#'  write_table_volume(connector_object,
#'     data,
#'     "my_table",
#'     overwrite = TRUE,
#'     tags = list("tag_name1" = "tag_value1")
#'  )
#' }
write_table_volume <- function(
  connector_object,
  x,
  name,
  overwrite = zephyr::get_option("overwrite", "connector.databricks"),
  tags = NULL
) {
  checkmate::assert_r6(
    x = connector_object,
    classes = "ConnectorDatabricksTable",
    null.ok = FALSE
  )
  checkmate::assert_character(x = name, null.ok = FALSE)
  checkmate::assert_logical(x = overwrite, null.ok = FALSE)
  checkmate::assert_list(tags, null.ok = TRUE)

  volume_name <- tmp_volume_name()
  temporary_volume <- tryCatch(
    {
      tmp_volume(connector_object, volume_name)
    },
    error = function(e) {
      zephyr::msg_danger(paste0(
        "Temporary volume creation failed. Error message:",
        e
      ))
    }
  )

  withr::defer({
    brickster::db_uc_volumes_delete(
      catalog = temporary_volume$catalog,
      schema = temporary_volume$schema,
      volume = volume_name
    )
  })

  zephyr::msg_info(
    "Writing to a table
    {connector_object$catalog}/{connector_object$schema}/{name}..."
  )

  # hms is not supported by parquet
  are_hms_variables_present <- any(purrr::map_lgl(x, hms::is_hms))
  if (are_hms_variables_present) {
    zephyr::msg_warning(
      "Some variables are in HMS format.
       This format is not supported by Databricks.\n
       We have to convert them to character."
    )
    x <- x |>
      dplyr::mutate(
        dplyr::across(dplyr::where(hms::is_hms), as.character)
      )
  }

  temporary_volume$write_cnt(
    x = x,
    name = paste0(name, ".parquet"),
    overwrite = overwrite
  )

  parquet_to_table(
    connector_object = connector_object,
    tmp_volume = temporary_volume,
    name = name,
    overwrite = overwrite
  )

  if (!is.null(tags)) {
    add_table_tags(
      connector_object = connector_object,
      tags = tags,
      name = name
    )
  }
  zephyr::msg_success("Table written successfully!")
}

#' List Databricks tables in a catalog based on tag values
#'
#' This function is used to list tables in a Databricks catalog based on a tag.
#'
#' @param connector_object [ConnectorDatabricksTable] object for interacting
#' with Databricks
#' @param tags String containing tag names and tag values in SQL query format,
#' e.g. `(tag_name1 = 'tag_value1' OR tag_name2 = 'tag_value2')`.
#' More info [here](https://docs.databricks.com/aws/en/database-objects/tags)
#' @return None
#' @noRd
#' @examples
#' \dontrun{
#' connector_object |>
#'     list_content_tags(tags = ("tag_value = 'value1'")))
#' }
list_content_tags <- function(
  connector_object,
  tags
) {
  checkmate::assert_r6(
    x = connector_object,
    classes = "ConnectorDatabricksTable",
    null.ok = FALSE
  )
  checkmate::assert_character(tags, null.ok = FALSE)

  warehouse_id <- brickster::db_sql_warehouse_list()[[1]]$id

  results <- execute_sql_query(
    glue::glue(
      "SELECT DISTINCT table_name
       FROM system.information_schema.table_tags
       WHERE
          catalog_name = '{connector_object$catalog}'
          AND schema_name = '{connector_object$schema}'
          AND {tags}
      "
    ),
    warehouse_id = warehouse_id
  )

  unique(unlist(results$result$data_array))
}

#' Read Databricks table into dataframe
#'
#' This function is used to read tables in a Databricks catalog with the option
#' of defining timepoint.
#'
#' More info [here](https://docs.databricks.com/gcp/en/delta/history#what-is-delta-lake-time-travel) # nolint
#'
#' @param connector_object [ConnectorDatabricksTable] object for interacting
#' with Databricks
#' @param name Table name
#' @param timepoint Timepoint in [Delta time travel syntax](https://docs.databricks.com/gcp/en/delta/history#delta-time-travel-syntax)  # nolint
#' format.
#' @param version Table version generated by the operation.
#' @return Data frame with values from the Databricks table
#' @noRd
#' @examples
#' \dontrun{
#' connector_object |>
#'     read_table_timepoint(name = "table_name",
#'                          timepoint = "2024-04-10 10:10:07 CEST"
#'      )
#' }
read_table_timepoint <- function(
  connector_object,
  name,
  timepoint = NULL,
  version = NULL
) {
  checkmate::assert_r6(
    x = connector_object,
    classes = "ConnectorDatabricksTable",
    null.ok = FALSE
  )
  checkmate::assert_character(name, null.ok = FALSE)
  checkmate::assert_numeric(version, null.ok = TRUE)

  sql_statement <- glue::glue(
    "SELECT * FROM {connector_object$catalog}.{connector_object$schema}.`{name}`" # nolint
  )

  if (!is.null(timepoint)) {
    timepoint <- as.POSIXct(timepoint)
    formatted_time <- format(timepoint, "%Y%m%d%H%M%OS3")
    formatted_time <- gsub("\\.", "", formatted_time)
    sql_statement <- glue::glue("{sql_statement}@{formatted_time}")
  } else if (!is.null(version)) {
    sql_statement <- glue::glue("{sql_statement}@v{version}")
  }

  warehouse_id <- brickster::db_sql_warehouse_list()[[1]]$id

  zephyr::msg_info(
    "Reading from a table
    {connector_object$catalog}.{connector_object$schema}.`{name}`..."
  )
  query_result <- execute_sql_query(
    query_string = sql_statement,
    warehouse_id = warehouse_id,
    catalog = connector_object$catalog,
    schema = connector_object$schema,
    disposition = "EXTERNAL_LINKS",
    format = "ARROW_STREAM"
  )

  if (query_result$manifest$total_chunk_count == 0) {
    zephyr::msg_success("Table read successfully!")
    return(DBI::dbGetQuery(connector_object$conn, sql_statement))
  }

  statement_id <- query_result$statement_id
  chunk_count <- query_result$manifest$total_chunk_count - 1

  results <- list()
  for (chunk in 0:chunk_count) {
    result <- brickster::db_sql_exec_result(
      statement_id = statement_id,
      chunk_index = chunk
    )

    results[[chunk + 1]] <- as.data.frame(arrow::read_ipc_stream(
      file = result$external_links[[1]]$external_link
    ))
  }

  zephyr::msg_success("Table read successfully!")

  dplyr::bind_rows(results)
}
