% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/io.R
\name{standardise_fdata}
\alias{standardise_fdata}
\title{Standardise reflections data}
\usage{
standardise_fdata(a, SG, hidx, Fobs = NULL, sigFobs = NULL,
  Fp = NULL, sigFp = NULL, Fm = NULL, sigFm = NULL,
  Phiobs = NULL, Phicalc = NULL)
}
\arguments{
\item{a}{Real numeric. Unit cell length in angstroms.}

\item{SG}{SG 2-letters character string. There are only two symmetries
possible when working within 1D crystallography, P1 (no 
symmetry) and P-1 (inversion through the origin). SG can be 
either "P1" or "P-1".}

\item{hidx}{Real numeric array. 1D unique (positive in the 1D context)
Miller indices.}

\item{Fobs}{Real numeric array. Amplitudes of structure factors.
If Fp and Fm are not NULL and Fobs is NULL, then Fobs are calculated
as averages of Fp and Fm. If both Fp, Fm and Fobs are included, input
Fobs are used, instead of Fp and Fm averages.}

\item{sigFobs}{Real numeric array. Errors associated with Fobs. If
sigFobs = NULL, errors are estimated from Fp and Fm. Default is NULL.}

\item{Fp}{Real numeric vector. Amplitudes of the positive 
component of Friedel (or Bijvoet) pairs (F+). Default is NULL,
i.e. no Fp included.}

\item{sigFp}{Real numeric vector. Errors associated with Fp. Default
is NULL, i.e. no sigFp included.}

\item{Fm}{Real numeric vector. Amplitudes of the negative 
component of Friedel (or Bijvoet) pairs (F-). Default is NULL,
i.e. no Fm included.}

\item{sigFm}{Real numeric vector. Errors associated with Fm. Default
is NULL, i.e. no sigFm included.}

\item{Phiobs}{Real numeric array. Phases (in degrees) of structure 
factors obtained with one of the methods used for structure solution.
Default is NULL.}

\item{Phicalc}{Real numeric array. Phases (in degrees) of structure 
factors calculated from the correct 1D structure. They are normally 
used to check correctness of Phiobs. Default is NULL.}
}
\value{
A named list with a variable number of elements. Some of them
 are always included; others are not:
 \itemize{
   \item{a    Real numeric. Unit cell length in angstroms. Always 
              included.}
   \item{SG.   Spacegroup 2-letters character string. There are only two 
               symmetries possible when working within 1D 
               crystallography, P1 (no symmetry) and P-1 (inversion 
               through the origin). SG can be either "P1" or "P-1". 
               Always included.}
   \item{hidx. Real numeric array. 1D unique (positive in the 1D context) 
               Miller indices. Always included.}
   \item{Fobs.      Real numeric array. Amplitudes of observed structure 
                    factors. Not always included.}
   \item{sigFobs.   Real numeric array. Errors associated with Fobs. Not 
                    always included.}
   \item{Fp.        Real numeric vector. Amplitudes of the positive 
                    component of Friedel (or Bijvoet) pairs (F+). Not
                    always included.}
   \item{sigFp.     Real numeric vector. Errors associated with Fp. 
                    Not always included.}
   \item{Fm.        Real numeric vector. Amplitudes of the negative 
                    component of Friedel (or Bijvoet) pairs (F-). Not always
                    included.}
   \item{sigFm.     Real numeric vector. Errors associated with Fm. Not
                    always included.}
   \item{Phiobs.    Real numeric array. Phases (in degrees) of structure 
                    factors obtained with one of the methods used for 
                    structure solution. Not always included.}
   \item{Phicalc.   Real numeric array. Phases (in degrees) of structure 
                    factors calculated from the correct 1D structure. 
                    They are normally used to check correctness of 
                    Phiobs. Not always included.}
         }
}
\description{
This function output a list with fields needed by most of the
functions dealing with structure factors. It is the equivalent
of the function \cite{standardise_sdata}, used to prepare atomic
structures data.
}
\examples{
# Create an arbitrary structure with a heavy atom (Fe)
a <- 20
SG <- "P1"
x0 <- c(1,2,6,16)
Z <- c(6,8,26,7)
B <- c(8,7,5,8)
occ <- c(1,1,1,1)
sdata <- standardise_sdata(a,SG,x0,Z,B,occ)

# Miller indices, from -5 to 5 (to include negatives for anomalous)
hidx <- -5:5

# Experimental structure factors with anomalous contribution
# (lambda = 1.74) for creating Fm and Fp. Errors only due to
# photons fluctuations.
set.seed(9195)   # For demo purposes.
ltmp <- sfobs(hidx,sdata,ntrialP=10,anoflag=TRUE,lbda=1.74)

# Fp and sigFp (Miller indices from 1 to 5)
isel <- 1:5
idx <- match(isel,hidx)
Fp <- ltmp$F[idx]
sigFp <- ltmp$sF[idx]

# Fm and sigFm
isel <- (-1):(-5)
idx <- match(isel,hidx)
Fm <- ltmp$F[idx]
sigFm <- ltmp$sF[idx]

# Now only positive Miller indices
hidx <- 1:5

# Create standardised data for reciprocal space
fdata <- standardise_fdata(a,SG,hidx,Fp=Fp,sigFp=sigFp,
         Fm=Fm,sigFm=sigFm)
         
# Fp and Fm
print(fdata$Fp)
print(fdata$sigFp)
print(fdata$Fm)
print(fdata$sigFm)

# Fobs and sigFobs automatically created
print(fdata$Fobs)
print(fdata$sigFobs)

# Structure factors without anomalous data for the same structure
hidx <- 1:5
set.seed(9195)   # For demo purposes.
ltmp <- sfobs(hidx,sdata,ntrialP=10)
Fobs <- ltmp$F
sigFobs <- ltmp$sF
fdata <- standardise_fdata(a,SG,hidx,Fobs=Fobs,sigFobs=sigFobs)
print(fdata)

}
