#' Load DAMS data from one or several continuous text files
#'
#' Uses "linked metadata" to load data from either single beam (DAM2) or multibeam (DAM5) arrays.
#'
#' @param metadata [data.table::data.table] used to load data (see detail)
#' @param date_format How dates are formatted in the DAM result files (see [read_dam_file])
#' @param FUN function (optional) to transform the data from each animal
#' immediately after is has been loaded.
#' @param ... extra arguments to be passed to `FUN`
#' @return A [behavr::behavr] table.
#' In addition to the metadata, it contains the data, whith the columns:
#' * `id` -- autogenerated unique identifier, one per animal
#' * `t` -- time
#' * `activity` -- number of beam crosses
#' @details
#' The linked metadata should be generated using [link_dam_metadata].
#' @examples
#' # This is where our toy data lives
#' root_dir <- damr_example_dir()
#'
#' # Metadata already made for us.
#' # It defines condition and genotype of each animal
#' data(single_file_metadata)
#' print(single_file_metadata)
#' # Linking:
#' metadata <- link_dam_metadata(single_file_metadata, root_dir)
#'
#' # We find and load the matching data
#' dt <- load_dam(metadata)
#' print(dt)
#' # An example of the use of FUN,
#' # we load only the first few reads by run `head()` on each animal,
#' # on the fly (no pun intended)
#' dt <- load_dam(metadata, FUN = head)
#' print(dt)
#' @seealso
#' * [behavr::behavr] --  the class of the resulting object
#' * [read_dam_file] --  to load data from a single file (without metadata)
#' @references
#' * [damr tutorial](https://rethomics.github.io/damr.html) -- how to use this function in practice
#' @aliases load_dam2
#' @export load_dam load_dam2
load_dam <- function(metadata, date_format="%d %b %y", FUN=NULL, ...){
  . = regions = start_datetime =  stop_datetime =  data = diff_t = NULL
  region_id = path = file_info =NULL


  tz="UTC"
  # TODO check for uniqueness in query!!
  q <- data.table::copy(metadata)
  q[, path:=sapply(file_info, function(x) x$path)]
  to_read <- q[,.(regions = list(region_id)),by=c("path","start_datetime","stop_datetime")]
  s <- to_read[,
               list(data=list(read_dam_file(path,
                                                    regions[[1]],
                                                    start_datetime,
                                                    stop_datetime,
                                                    tz=tz,
                                                    date_format=date_format)
                              )
                    ),
               by=c("path","start_datetime","stop_datetime")]

  d <- behavr::bind_behavr_list(s[,data])

  # new metadata had the columns in metadata that are not in d[metadata] already
  met <- d[meta=T]
  met <- met[
    metadata[,
      c("id", setdiff(colnames(metadata), colnames(met))),
      with=F],
    on="id"]

  data.table::setkeyv(met,"id")
  # replace metadata

  behavr::setmeta(d, met)
  if(!is.null(FUN))
    d <- d[,FUN(.SD, ...),by="id"]
  d
}


load_dam2 <- function(metadata, FUN=NULL, ...){
  message("load_dam2 is deprecated, please use load_dam instead")
  load_dam(metadata, FUN=NULL, ...)
}
