% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\name{formula.fixest}
\alias{formula.fixest}
\alias{formula.fixest_multi}
\title{Extract the formula of a \code{fixest} fit}
\usage{
\method{formula}{fixest}(x, type = "full", fml.update = NULL, fml.build = NULL, ...)

\method{formula}{fixest_multi}(x, type = "full", fml.update = NULL, fml.build = NULL, ...)
}
\arguments{
\item{x}{An object of class \code{fixest}. Typically the result of a \code{\link{femlm}}, \code{\link{feols}}
or \code{\link{feglm}} estimation.}

\item{type}{A character scalar. Default is \code{type = "full"} which gives back a formula
containing the linear part of the model along with the fixed-effects (if any) and the
IV part (if any). Here is a description of the other types:
\itemize{
\item \code{full.noiv}: the full formula without the IV part
\item \code{full.nofixef.noiv}: the full formula without the IV nor the fixed-effects part
\item \code{lhs}: a one-sided formula with the dependent variable
\item \code{rhs}: a one-sided formula of the right hand side without the IVs (if any)
\item \code{rhs.nofixef} or \code{indep}: a one-sided formula of the right hand side without the
fixed-effects nor IVs (if any), it is equivalent to the
independent variables
\item \code{NL}: a one-sided formula with the non-linear part (if any)
\item \code{fixef}: a one-sided formula containing the fixed-effects
\item \code{iv}: a two-sided formula containing the endogenous variables (left) and the
instruments (right)
\item \code{iv.endo}: a one-sided formula of the endogenous variables
\item \code{iv.inst}: a one-sided formula of the instruments
\item \code{iv.reduced}: a two-sided formula representing the reduced form,
that is \code{y ~ exo + inst}
}}

\item{fml.update}{A formula representing the changes to be made to the original
formula. By default it is \code{NULL}.
Use a dot to refer to the previous variables in the current part.
For example: \code{. ~ . + xnew} will add the variable \code{xnew} as an explanatory variable.
Note that the previous fixed-effects (FEs) and IVs are implicitly forwarded.
To rerun without the FEs or the IVs, you need to set them to 0 in their respective slot.
Ex, assume the original formula is: \code{y ~ x | fe | endo ~ inst}, passing \code{. ~ . + xnew}
to fml.update leads to \code{y ~ x + xnew | fe | endo ~ inst} (FEs and IVs are forwarded).
To add xnew and remove the IV part: use \code{. ~ . + xnew | . | 0} which leads to
\code{y ~ x + xnew | fe}.}

\item{fml.build}{A formula or \code{NULL} (default). You can create a new formula based
on the parts of the formula of the object in \code{x}. In this argument you have access
to these specific variables:
\itemize{
\item \code{.}: to refer to the part of the original formula
\item \code{.lhs}: to refer to the dependent variable
\item \code{.indep}: to refer to the independent variables (excluding the fixed-effects)
\item \code{.fixef}: to refer to the fixed-effects
\item \code{.endo}: to refer to endogenous variables in an IV estimation
\item \code{.inst}: to refer to instruments in an IV estimation
}

Example, the original estimation was \code{y ~ x1 | z ~ inst}. Then
\code{fml.build = . ~ .endo + .} leads to \code{y ~ z + x1}.}

\item{...}{Not currently used.}
}
\value{
It returns either a one-sided formula, either a two-sided formula.
}
\description{
This function extracts the formula from a \code{fixest} estimation (obtained with \code{\link{femlm}},
\code{\link{feols}} or \code{\link{feglm}}). If the estimation was done with fixed-effects, they are added
in the formula after a pipe (\dQuote{|}). If the estimation was done with a non
linear in parameters part, then this will be added in the formula in between \code{I()}.
}
\details{
The arguments \code{type}, \code{fml.update} and \code{fml.build} are exclusive: they
cannot be used at the same time.
}
\examples{

# example estimation with IVS and FEs
base = setNames(iris, c("y", "x1", "endo", "instr", "species"))
est = feols(y ~ x1 | species | endo ~ instr, base)

# the full formula
formula(est)

# idem without the IVs nor the FEs
formula(est, "full.nofixef.noiv")

# the reduced form
formula(est, "iv.reduced")

# the IV relation only
formula(est, "iv")

# the dependent variable => onse-sided formula
formula(est, "lhs")

# using update, we add x1^2 as an independent variable:
formula(est, fml.update = . ~ . + x1^2)

# using build, see the difference => the FEs and the IVs are not inherited
formula(est, fml.build = . ~ . + x1^2)

# we can use some special variables
formula(est, fml.build = . ~ .endo + .indep)


}
\seealso{
See also the main estimation functions \code{\link{femlm}}, \code{\link{feols}} or \code{\link{feglm}}.
\code{\link{model.matrix.fixest}}, \code{\link{update.fixest}}, \code{\link{summary.fixest}}, \code{\link{vcov.fixest}}.
}
\author{
Laurent Berge
}
