.onAttach <- function(libname, pkgname) {
  startup <- function() {
    cli::cli_inform(c(
      "v" = "{.strong {.pkg invivoPKfit} v{utils::packageVersion('invivoPKfit')} has been successfully loaded!}",
      "i" = paste("For help, please run:", cli::col_cyan("help('invivoPKfit')"))
    ), class = "packageStartupMessage"
    )
  }

  startup()
}





#' Creating a simple test CvT dataset
#'
#' @param params A named numeric vector of model parameter values.
#' @param time A numeric vector of times, reflecting the time point when
#'  concentration is measured after the corresponding single bolus dose. Must be
#'  same length as `dose` and `iv.dose`, or length 1.
#' @param dose A numeric vector of doses, reflecting single bolus doses
#'  administered at time 0. Must be same length as `time` and `iv.dose`, or
#'  length 1. In this model, it is expected that this value represents a measurement
#'  of radioactive particles from a radiolabeling experiment.
#' @param route A character vector, reflecting the route of administration of
#'  each single bolus dose: `'oral'` or `'iv'`.  Must be same length as `time`
#'  and `dose`, or length 1.
#' @param medium A character vector reflecting the medium in which each resulting
#'  concentration is to be calculated: "blood" or "plasma". Default is "plasma".
#'  Must be same length as `time` and `dose`, or length 1.
#' @param N Numeric, positive and non-zero integer. Number of individual subjects.
#' @param var Numeric between 0 and 1. Describes variation in the measurements.
#'
#' @return A data frame with concentration over time data.

pseudo_cvt <- function(
    params = c(
      Clint = 10,
      Q_gfr = 0.31,
      Q_totli = 0.743,
      Fup = 0.2,
      Vdist = 1.2,
      Fgutabs = 0.75,
      kgutabs = 0.3,
      Rblood2plasma = 0.8,
      Frec = 0.95
    ),
    time = seq(0, 30, by = 0.5),
    dose = 100,
    route = c("oral", "iv"),
    medium = c("blood", "plasma", "excreta"),
    N = 4,
    var = 0.5) {

  n_routes <- length(route)
  n_media <- length(medium)
  n_subjects <- N
  timepoints <- time
  n_timepoints <- length(timepoints)
  total_combinations <- n_routes * n_media * n_subjects * n_timepoints

  if (total_combinations > 10000) {
    if (n_subjects > 5) {
      n_subjects <- 3
      message("Setting subjects to 3.")
    }

    ideal_timepoints <- 10000 %/% (n_media * n_subjects * n_routes)
    min_tp <- min(time)
    max_tp <- max(time)
    timepoints <- seq(min_tp, max_tp, length.out = ideal_timepoints)
    message("Reduced number of timepoints to ", ideal_timepoints)
  }


  tmp_subject_time <- data.frame(
    Time = rep(time, each = n_subjects),
    Subject_ID = rep(seq(1, n_subjects), n_timepoints)
  )

  tmp_route_media <- data.frame(
    Route = rep(route, each = n_media),
    Medium = rep(medium, n_routes)
  )

  tmp_cvt <- tidyr::crossing(tmp_subject_time, tmp_route_media)

  tmp_cvt$Conc <- with(
    tmp_cvt,
    cp_1comp_rad(params, time = Time, dose,
                 route = Route, medium = Medium)
  )

  tmp_cvt <- tmp_cvt |>
    dplyr::rowwise() |>
    dplyr::mutate(Conc = pmin(dose,
                              pmax(sqrt(.Machine$double.eps),
                                   rnorm(1, mean = Conc, sd = var*log2(dose))
                              )
    )) |>
    ungroup()

  return(tmp_cvt)
}


#' Creates a [cli::cli_fmt()] output for the pattern "name(x), length = "
#'
#' @param x A named vector with a single value for each element.
#' @param extra A character vector that should be printed between each name and value.
#' @return A cli::cli_fmt output list.
#' @author Gilberto Padilla Mercado
#'

post_name_value <- function(x, extra = "") {
  cli::cli_fmt({
    cli::cli_ul()
    for (k in seq_along(x)) {
      cli::cli_li(paste("{names(x[k])},", extra, "{x[k]}"))
    }
    cli::cli_end()
  })
}


