% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diary_functions.R
\name{icc_11}
\alias{icc_11}
\title{Intraclass Correlation for Multilevel Analysis: ICC(1,1)}
\usage{
icc_11(x, grp, how = "lme", REML = TRUE)
}
\arguments{
\item{x}{numeric vector.}

\item{grp}{atomic vector the same length as \code{x} providing the grouping
variable.}

\item{how}{character vector of length 1 specifying how the ICC(1,1) should be
calculated. There are four options: 1) "lme" uses a linear mixed effects
model with the function \code{\link[nlme]{lme}} from the package
\code{nlme}, 2) "lmer" uses a linear mixed effects modeling with the
function \code{\link[lme4]{lmer}} from the package \code{lme4}, 3) "aov"
uses a one-way analysis of variance with the function
\code{\link[stats]{aov}}, and 4) "raw" uses the observed variances, which
provides a biased estimate of the ICC(1,1) and is not recommended (It is
only included for teaching purposes).}

\item{REML}{logical vector of length 1 specifying whether restricted maximum
likelihood estimation (TRUE) should be used rather than traditional maximum
likelihood estimation (FALSE). Only used for linear mixed effects models if
how = "lme" or how = "lmer".}
}
\value{
numeric vector of length 1 providing ICC(1,1) and computed based on
  the \code{how} argument.
}
\description{
\code{icc_11} computes the intraclass correlation (ICC) based on a single
rater with a single dimension, aka ICC(1,1). Traditionally, this is the type
of ICC used for multilevel analysis where the value is interpreted as the
proportion of variance accounted for by group membership. In other words,
ICC(1,1) = the proportion of between-group variance; 1 - ICC(1,1) = the
proportion of within-group variance.
}
\examples{

# BALANCED DATA (how = "aov" and "lme"/"lmer" do YES provide the same value)

str(InsectSprays)
icc_11(x = InsectSprays$"count", grp = InsectSprays$"spray", how = "aov")
icc_11(x = InsectSprays$"count", grp = InsectSprays$"spray", how = "lme")
icc_11(x = InsectSprays$"count", grp = InsectSprays$"spray", how = "lmer")
icc_11(x = InsectSprays$"count", grp = InsectSprays$"spray",
   how = "raw") # biased estimator and not recommended. Only available for teaching purposes.

# UN-BALANCED DATA (how = "aov" and "lme"/"lmer" do NOT provide the same value)

dat <- as.data.frame(lmeInfo::Bryant2016)
icc_11(x = dat$"outcome", grp = dat$"case", how = "aov")
icc_11(x = dat$"outcome", grp = dat$"case", how = "lme")
icc_11(x = dat$"outcome", grp = dat$"case", how = "lmer")
icc_11(x = dat$"outcome", grp = dat$"case", how = "lme", REML = FALSE)
icc_11(x = dat$"outcome", grp = dat$"case", how = "lmer", REML = FALSE)

# how = "lme" does not account for any correlation structure
lme_obj <- nlme::lme(outcome ~ 1, random = ~ 1 | case,
   data = dat, na.action = na.exclude,
   correlation = nlme::corAR1(form = ~ 1 | case), method = "REML")
var_corr <- nlme::VarCorr(lme_obj) # VarCorr.lme
vars <- as.double(var_corr[, "Variance"])
btw <- vars[1]
wth <- vars[2]
btw / (btw + wth)

}
\seealso{
\code{\link{iccs_11}} # ICC(1,1) for multiple variables,
   \code{\link{icc_all_by}} # all six types of ICCs by group,
   \code{\link[nlme]{lme}} # how = "lme" function,
   \code{\link[lme4]{lmer}} # how = "lmer" function,
   \code{\link[stats]{aov}} # how = "aov" function,
}
