% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rolr-functions.R
\name{rsolr12}
\alias{rsolr12}
\title{Finding Optimal Cutpoints Using Simple Ordered Logrank (SOL) Tests}
\usage{
rsolr12(times, status, x, ns = 15, alt = "decrease",
  method = "approximate")
}
\arguments{
\item{times}{Survival outcome.}

\item{status}{Censoring indicator which takes 1 when an event occurs at end
of study and 0 otherwise.}

\item{x}{A continuous covariate.}

\item{ns}{Minimum number of subjects in each group after dichotomizing the
covariate.}

\item{alt}{A character that takes either \code{"decrease"} or \code{"increase"}
to represent a positive or negative association between the covariate
and survival.}

\item{method}{A character that takes either \code{"approximate"} or
\code{"exact"} where an approximate or exact method will be used.}
}
\value{
Returns a list with three elements, the first one being the test
  statistics for all cutpoints considered (except the first and last \code{ns}
  points), and the second and third elements being the best splits obtained
  from using the SOL-1 and SOL-2 tests.
}
\description{
Using two simple ordered logrank tests (SOL-1 and SOL-2), the
  \code{rsolr12} function finds two optimal cutpoints to divide the entire
  dataset into three groups based on a continuous covariate and a survival
  outcome. It is a fast procedure that makes use of the running logrank test
  (\code{\link{rlr}}) to improve on computing speed.
}
\details{
When the association is positive, that is, larger covariate
  values leading to worse survival, and you enter \code{alt = "decrease"}, the test
  statistics will be positive, but if you enter \code{trend = "increase"} the
  test statistics will be negative. Opposite is true when the association
  is negative. You want to make sure to enter the option so that the
  resulting test statistics are positive.
}
\examples{
library(rolr)

##### -------- Example 1

#simulate data with 2 underlying true cutpoints and hazard goes up as x goes up
d=simdata(nn = 150, hr = c(1, 2, 3), hazard.func = "step",
          props=c(1/3, 1/3, 1/3), censoring.rate = 0)

#using alt = 'decrease', the test statistics are positive, so it is good
res=rsolr12(times=d$times, status=d$status, x=d$x, ns=15, alt='decrease')
names(res)
res[['best.splits.solr1']]
res[['best.splits.solr2']]

#do it again using alt = 'increase', now the test statistics are negative and
#so the results are not right. So you have to switch back to alt='decrease' to
#get positive statistics and the correct optimal cutpoints here.
res2=rsolr12(times=d$times, status=d$status, x=d$x, ns=15, alt='increase')
res2[['best.splits.solr1']]
res2[['best.splits.solr2']]

##### -------- Example 2

#simulate data with true cutpoints and hazard goes down as covariate goes up
d=simdata(nn = 150, hr = c(3, 2, 1), hazard.func = "step",
          props=c(1/3, 1/3, 1/3), censoring.rate = 0)

#using alt = 'decrease', the test statistics are negative (so the results
#are not right).
res=rsolr12(times=d$times, status=d$status, x=d$x, ns=15, alt='decrease')
res[['best.splits.solr1']]
res[['best.splits.solr2']]

#do it again using alt = 'increase', now it is right
res2=rsolr12(times=d$times, status=d$status, x=d$x, ns=15, alt='increase')
res2[['best.splits.solr1']]
res2[['best.splits.solr2']]
}
\references{
See main package help page.
}
\seealso{
\code{\link{rmolr}}, \code{\link{rhier}}
}

