#' This class represents an SQL logical operator.
#'
#' Used to generate SQL expressions involving a commutative binary operator like
#' in "a + 10 + b".
#'
#' @examples
#' # To generate "a + 10 + b":
#' ExprCommOp$new("+", list(ExprField$new("a"), ExprValue$new(10),
#'                          ExprField$new("b")))
#'
#' @import R6
#' @include ExprComp.R
#' @export
ExprCommOp <- R6::R6Class("ExprCommOp",
  inherit = ExprComp,
  public = list(

    #' @description
    #' Initializer.
    #' @param op The logical operator, as a string.
    #' @param expr A list of logical expressions.
    #' @return Nothing.
    initialize = function(op, expr = NULL) {
      super$initialize()
      chk::chk_string(op)
      private$op <- op
      if (is.null(expr)) {
        private$expr <- list()
      } else {
        chk::chk_all(expr, chk::chk_is, "Expr")
        private$expr <- expr
      }

      return(invisible(NULL))
    },

    #' @description
    #' Add an SQL expression to the logical operator.
    #' @param expr A Expr instance.
    #' @return Nothing.
    add = function(expr) {
      chk::chk_is(expr, "Expr")
      private$expr <- c(private$expr, expr)

      return(invisible(NULL))
    },

    #' @description
    #' Returns the number of expressions.
    #' @return The number of expressions in this logical operator.
    nb_expr = function() {
      return(length(private$expr))
    },

    #' @description
    #' Generates the list of tokens representing this statement.
    #' @return A list of Token objects.
    getTokens = function() {
      tokens <- list()
      if (length(private$expr) > 0L) {
        if (private$paren) {
          tokens <- c(tokens, .lparen)
        }
        i <- 0L
        for (e in private$expr) {
          if (i > 0L) {
            tokens <- c(tokens, make_middle_op(private$op))
          }
          tokens <- c(tokens, e$getTokens())
          i <- i + 1L
        }
        if (private$paren) {
          tokens <- c(tokens, .rparen)
        }
      }
      return(tokens)
    }
  ),

  private = list(
    op = NULL,
    expr = NULL
  )
)
