#' @title Operating Characteristics Function (TITE Method)
#' @description Compute operating characteristics for a stopping rule at a set of toxicity rates. Characteristics calculated include the overall rejection probability, the expected number of patients evaluated, and the expected number of events.
#'
#' @param rule A \code{rule.tite} object with the safety stopping rule for evaluation
#' @param ps Vector of cumulative incidence probabilities for toxicity at time \code{tau}
#' @param ps.compt Vector of cumulative incidence probabilities for competing risks at time \code{tau} if competing risks are involved. Set to NULL if no competing risks exist.
#' @param tau Length of observation period
#' @param MC Number of Monte Carlo replicated datasets to simulate
#' @param A Length of accrual period
#' @param family Event time distribution, choices including power distribution ('power') and Weibull distribution ('weibull')
#' @param s Shape parameter for Weibull distribution or power parameter for power distribution
#'
#'
#' @return A matrix with columns containing the toxicity probabilities \code{ps}, competing risk probability (0 for survival outcome), the corresponding rejection probabilities, and the corresponding expected number of events. If \code{tau} and \code{A} are also specified, the expected numbers of enrolled patients and the expected calendar time at the point of stopping/study end are also included.
#'
#' @export
#'
#' @details
#' Times are generated for each event cause so that its marginal distribution follows a Weibull or power family distribution as specified by the user.
#'
#' For the Weibull distribution, the cumulative distribution function is \eqn{1- \exp(- \lambda t^s), t \ge 0}, where \eqn{\lambda} is the rate parameter and \code{s} is the shape parameter.
#'
#' The power distribution has the cumulative distribution function \eqn{k (t /\tau)^s, 0 \le t \le \tau k^{-1/s}}, where \code{s} is the power parameter and \eqn{k} is the value at \eqn{t=\tau}.
#'
#' For the toxicity event distribution, the Weibull rate parameter is \eqn{\lambda = - \log(1-p) / \tau} and the power parameter is \eqn{k = p}, where \eqn{p} is the cumulative incidence of toxicity at \eqn{t=\tau}. For competing risk events' distribution, \eqn{\lambda = - 1 / \tau \log(1-p_c)} and \eqn{k = p_c}, where \eqn{p_c} is the cumulative incidence of competing events at \eqn{t=\tau}.
#'
#' @examples
#' \dontrun{
#' # Bayesian beta-extended binomial method in 50 patient cohort at 10% level,
#' # expected toxicity probability of 20%
#' bb_rule = calc.rule.tite(n=50,p0=0.20,alpha=0.10,type="BB",param=c(2,8))
#'
#' # Compute operating characteristics at toxicity probabilities of 20%, 25%, 30%, 35%, and 40%
#' OC.rule.tite(rule=bb_rule,ps=seq(0.2,0.4,0.05), MC =1000, tau=30,A=730, family = 'weibull', s = 2)
#'}

OC.rule.tite = function (rule, ps, ps.compt=NULL, MC, tau, A, family="power", s=1) {
  if (MC >= 0) {
    ps = cbind(ps,ps.compt)
    if (is.null(ps.compt)){
      tab = matrix(0, nrow = length(ps), ncol = 6)
      for (i in 1:length(ps)){
        op = opchars.tite(rule = rule, p = ps[i,], tau = tau, MC = MC, A = A, family = family, s = s)
        tab[i, ] = c(ps[i], 0, op$Reject.prob, op$Exp.tox, op$Exp.n, op$Exp.caltime)
      }
    } else {
      tab = matrix(0, nrow = nrow(ps), ncol = 6)
      for (i in 1:nrow(ps)){
        op = opchars.tite(rule = rule, p = ps[i,], tau = tau, MC = MC, A = A, family = family, s = s)
        tab[i, ] = c(ps[i,1], ps[i,2], op$Reject.prob, op$Exp.tox, op$Exp.n, op$Exp.caltime)
      }
    }
    colnames(tab) = c("p", "p.compt","Reject Prob", "E(Events)",
                      "E(Enrolled)", "E(Calendar time)")
    return(tab)
  }
  else {
    print("Error: MC must be a nonnegative number")
  }
}
