testthat::describe("report_load_srv", {
  it("loading reporter restores saved content", {
    testthat::skip_if_not_installed("ggplot2")

    card <- teal.reporter::teal_card(
      "## Header 2 text",
      "A paragraph of default text",
      ggplot2::ggplot(iris, ggplot2::aes(x = Petal.Length)) +
        ggplot2::geom_histogram(binwidth = 0.2)
    )

    reporter <- teal.reporter::Reporter$new()$set_id("xyz")
    reporter$append_cards(list(card))
    reporter_path <- reporter$to_jsondir(withr::local_tempdir())

    temp_zip_file <- tempfile(pattern = "report_", fileext = ".zip")
    zip::zipr(temp_zip_file, reporter_path)

    shiny::testServer(
      report_load_srv,
      args = list(reporter = reporter),
      expr = {
        reporter$reset()
        session$setInputs(`reporter_load` = 0)
        session$setInputs(
          archiver_zip = list(datapath = temp_zip_file, name = basename(temp_zip_file))
        )
        session$setInputs(`reporter_load_main` = 0)
        testthat::expect_length(reporter$get_cards(), 1)
        testthat::expect_length(reporter$get_blocks(), 4)

        testthat::expect_match(reporter$get_blocks()[[1]], "# .*") # Title is added automatically
        testthat::expect_identical(reporter$get_blocks()[[2]], "## Header 2 text")
        testthat::expect_identical(reporter$get_blocks()[[3]], "A paragraph of default text")
        testthat::expect_s3_class(reporter$get_blocks()[[4]], "recordedplot")
      }
    )
  })

  it("loading reporter with ReportCard restores saved content (with old blocks)", {
    testthat::skip_if_not_installed("ggplot2")

    card <- teal.reporter::ReportCard$new()
    card$append_text("Header 2 text", "header2")
    card$append_text("A paragraph of default text", "header3")
    card$append_plot(
      ggplot2::ggplot(iris, ggplot2::aes(x = Petal.Length)) +
        ggplot2::geom_histogram(binwidth = 0.2)
    )

    reporter <- teal.reporter::Reporter$new()$set_id("xyz")
    reporter$append_cards(list(card))
    reporter_path <- reporter$to_jsondir(withr::local_tempdir())

    temp_zip_file <- tempfile(pattern = "report_", fileext = ".zip")
    zip::zipr(temp_zip_file, reporter_path)

    shiny::testServer(
      report_load_srv,
      args = list(reporter = reporter),
      expr = {
        reporter$reset()
        session$setInputs(`reporter_load` = 0)
        session$setInputs(
          archiver_zip = list(datapath = temp_zip_file, name = basename(temp_zip_file))
        )
        session$setInputs(`reporter_load_main` = 0)
        testthat::expect_length(reporter$get_cards(), 1)
        testthat::expect_length(reporter$get_blocks(), 4)
        testthat::expect_type(reporter$get_blocks()[[2]], "character")
        testthat::expect_identical(reporter$get_blocks()[[2]], "## Header 2 text")
        testthat::expect_type(reporter$get_blocks()[[3]], "character")
        testthat::expect_identical(reporter$get_blocks()[[3]], "### A paragraph of default text")
        testthat::expect_s3_class(reporter$get_blocks()[[4]], "recordedplot")
      }
    )
  })

  it("fail to load a reporter because of different id", {
    reporter <- teal.reporter::Reporter$new()
    reporter$set_id("xyz")

    reporter_path <- reporter$to_jsondir(withr::local_tempdir())
    temp_zip_file <- withr::local_tempfile(pattern = "report_", fileext = ".zip")
    zip::zipr(temp_zip_file, reporter_path)

    reporter <- teal.reporter::Reporter$new()$set_id("different")

    oo <- capture_output(shiny::testServer(
      report_load_srv,
      args = list(reporter = reporter),
      expr = {
        reporter$reset()
        session$setInputs(`reporter_load` = 0)
        session$setInputs(
          archiver_zip = list(
            datapath = temp_zip_file,
            name = basename(temp_zip_file)
          )
        )
        session$setInputs(`reporter_load_main` = 0)
      }
    ))
    testthat::expect_true(grepl("Loaded Report id has to match the current instance one", oo))
  })
})
