#' Power Calculation for Two Co-Primary Binary Endpoints (Approximate)
#'
#' Calculates the power for a two-arm superiority trial with two co-primary
#' binary endpoints using various asymptotic normal approximation methods, as
#' described in Sozu et al. (2010).
#'
#' @param n1 Sample size for group 1 (test group)
#' @param n2 Sample size for group 2 (control group)
#' @param p11 True probability of responders in group 1 for the first outcome (0 < p11 < 1)
#' @param p12 True probability of responders in group 1 for the second outcome (0 < p12 < 1)
#' @param p21 True probability of responders in group 2 for the first outcome (0 < p21 < 1)
#' @param p22 True probability of responders in group 2 for the second outcome (0 < p22 < 1)
#' @param rho1 Correlation between the two outcomes for group 1
#' @param rho2 Correlation between the two outcomes for group 2
#' @param alpha One-sided significance level (typically 0.025 or 0.05)
#' @param Test Statistical testing method. One of:
#'   \itemize{
#'     \item \code{"AN"}: Asymptotic normal method without continuity correction
#'     \item \code{"ANc"}: Asymptotic normal method with continuity correction
#'     \item \code{"AS"}: Arcsine method without continuity correction
#'     \item \code{"ASc"}: Arcsine method with continuity correction
#'   }
#'
#' @return A data frame with the following columns:
#'   \item{n1}{Sample size for group 1}
#'   \item{n2}{Sample size for group 2}
#'   \item{p11, p12, p21, p22}{Response probabilities}
#'   \item{rho1, rho2}{Correlations}
#'   \item{alpha}{One-sided significance level}
#'   \item{Test}{Testing method used}
#'   \item{power1}{Power for the first endpoint alone}
#'   \item{power2}{Power for the second endpoint alone}
#'   \item{powerCoprimary}{Power for both co-primary endpoints}
#'
#' @details
#' This function implements four approximate power calculation methods:
#'
#' \strong{Asymptotic Normal (AN):} Uses the standard normal approximation without
#' continuity correction (equations 3-4 in Sozu et al. 2010).
#'
#' \strong{Asymptotic Normal with Continuity Correction (ANc):} Includes Yates's
#' continuity correction (equation 5 in Sozu et al. 2010).
#'
#' \strong{Arcsine (AS):} Uses arcsine transformation without continuity correction
#' (equation 6 in Sozu et al. 2010).
#'
#' \strong{Arcsine with Continuity Correction (ASc):} Arcsine method with continuity
#' correction (equation 7 in Sozu et al. 2010).
#'
#' The correlation between test statistics for the two endpoints depends on the method:
#'
#' \strong{For AN and ANc methods:}
#' \deqn{\rho_{nml} = \frac{\sum_{j=1}^{2} \rho_j \sqrt{\nu_{j1}\nu_{j2}}/n_j}
#'       {se_1 \times se_2}}
#' where \eqn{\nu_{jk} = p_{jk}(1-p_{jk})}.
#'
#' \strong{For AS method:}
#' \deqn{\rho_{arc} = \frac{n_2 \rho_1 + n_1 \rho_2}{n_1 + n_2}}
#' This is the weighted average of the correlations from both groups.
#'
#' \strong{For ASc method:}
#' \deqn{\rho_{arc,c} = \frac{1}{se_1 \times se_2}
#'       \left(\frac{\rho_1 \sqrt{\nu_{11}\nu_{12}}}{4n_1\sqrt{\nu_{11,c}\nu_{12,c}}} +
#'       \frac{\rho_2 \sqrt{\nu_{21}\nu_{22}}}{4n_2\sqrt{\nu_{21,c}\nu_{22,c}}}\right)}
#' where \eqn{\nu_{jk,c} = (p_{jk} + c_j)(1 - p_{jk} - c_j)},
#' \eqn{c_1 = -1/(2n_1)}, and \eqn{c_2 = 1/(2n_2)}.
#'
#' The correlation bounds are automatically checked using \code{\link{corrbound2Binary}}.
#'
#' @references
#' Sozu, T., Sugimoto, T., & Hamasaki, T. (2010). Sample size determination in
#' clinical trials with multiple co-primary binary endpoints. \emph{Statistics in
#' Medicine}, 29(21), 2169-2179.
#'
#' @examples
#' # Power calculation using asymptotic normal method
#' power2BinaryApprox(
#'   n1 = 200,
#'   n2 = 100,
#'   p11 = 0.5,
#'   p12 = 0.4,
#'   p21 = 0.3,
#'   p22 = 0.2,
#'   rho1 = 0.7,
#'   rho2 = 0.7,
#'   alpha = 0.025,
#'   Test = 'AN'
#' )
#'
#' # Power calculation with continuity correction
#' power2BinaryApprox(
#'   n1 = 200,
#'   n2 = 100,
#'   p11 = 0.5,
#'   p12 = 0.4,
#'   p21 = 0.3,
#'   p22 = 0.2,
#'   rho1 = 0.7,
#'   rho2 = 0.7,
#'   alpha = 0.025,
#'   Test = 'ANc'
#' )
#'
#' # Power calculation using arcsine method
#' power2BinaryApprox(
#'   n1 = 150,
#'   n2 = 150,
#'   p11 = 0.6,
#'   p12 = 0.5,
#'   p21 = 0.4,
#'   p22 = 0.3,
#'   rho1 = 0.5,
#'   rho2 = 0.5,
#'   alpha = 0.025,
#'   Test = 'AS'
#' )
#'
#' @export
#' @importFrom pbivnorm pbivnorm
#' @importFrom stats qnorm
power2BinaryApprox <- function(n1, n2, p11, p12, p21, p22, rho1, rho2, alpha, Test) {

  # Check that rho1 is within valid bounds
  bounds1 <- corrbound2Binary(p11, p12)
  if (rho1 < bounds1[1] | rho1 > bounds1[2]) {
    stop(paste0("rho1 must be within [", round(bounds1[1], 4), ", ",
                round(bounds1[2], 4), "]"))
  }

  # Check that rho2 is within valid bounds
  bounds2 <- corrbound2Binary(p21, p22)
  if (rho2 < bounds2[1] | rho2 > bounds2[2]) {
    stop(paste0("rho2 must be within [", round(bounds2[1], 4), ", ",
                round(bounds2[2], 4), "]"))
  }

  # Standard normal quantiles
  z_alpha <- qnorm(1 - alpha)

  # Define common variance parameters
  nu_1k <- c(p11, p12) * (1 - c(p11, p12))
  nu_2k <- c(p21, p22) * (1 - c(p21, p22))

  if (Test == "AN" | Test == "ANc") {

    # Calculate pooled proportions for each endpoint
    p_k <- (n1 * c(p11, p12) + n2 * c(p21, p22)) / (n1 + n2)

    # Calculate treatment effects
    delta_k <- c(p11, p12) - c(p21, p22)

    # Calculate standard errors under null and alternative hypotheses
    se_k0 <- sqrt((1 / n1 + 1 / n2) * p_k * (1 - p_k))
    se_k <- sqrt(nu_1k / n1 + nu_2k / n2)

    # Calculate correlation between test statistics (equation 4 in Sozu et al. 2010)
    rho <- '+'(
      rho1 * sqrt(prod(nu_1k)) / n1,
      rho2 * sqrt(prod(nu_2k)) / n2
    ) / prod(se_k)

    if (Test == "AN") {

      # Asymptotic normal method without continuity correction

      # Critical values
      c_val <- (delta_k - se_k0 * z_alpha) / se_k

    } else if (Test == "ANc") {

      # Asymptotic normal method with Yates's continuity correction

      # Calculate continuity correction
      ycc <- (1 / 2) * (1 / n1 + 1 / n2)

      # Critical values
      c_val <- (delta_k - se_k0 * z_alpha - ycc) / se_k
    }

  } else if (Test == "AS" | Test == "ASc") {

    # Standard error for arcsine transformation
    se <- (1 / 2) * sqrt(1 / n1 + 1 / n2)

    if (Test == "AS") {

      # Arcsine method without continuity correction

      # Apply arcsine transformation to probabilities
      delta_k <- asin(sqrt(c(p11, p12))) - asin(sqrt(c(p21, p22)))

      # Critical values
      c_val <- delta_k / se - z_alpha

      # Calculate correlation between test statistics for arcsine method
      rho <- (n2 * rho1 + n1 * rho2) / (n1 + n2)

    } else if (Test == "ASc") {

      # Arcsine method with continuity correction

      # Define continuity correction constants
      c1 <- -1 / (2 * n1)
      c2 <-  1 / (2 * n2)

      # Apply arcsine transformation with continuity correction
      delta_k <- '-'(
        asin(sqrt(c(p11, p12) + c1)),
        asin(sqrt(c(p21, p22) + c2))
      )

      # Calculate adjusted variances for continuity correction
      nu_1k_c <- (c(p11, p12) + c1) * (1 - c(p11, p12) - c1)
      nu_2k_c <- (c(p21, p22) + c2) * (1 - c(p21, p22) - c2)
      se_k <- sqrt(nu_1k / (4 * n1 * nu_1k_c) + nu_2k / (4 * n2 * nu_2k_c))

      # Critical values
      c_val <- (delta_k - se * z_alpha) / se_k

      # Calculate correlation between test statistics with continuity correction
      rho <- '*'(
        1 / prod(se_k),
        '+'(
          rho1 * sqrt(prod(nu_1k)) / (4 * n1 * sqrt(prod(nu_1k_c))),
          rho2 * sqrt(prod(nu_2k)) / (4 * n2 * sqrt(prod(nu_2k_c)))
        )
      )

    }
  }

  # Calculate power for individual endpoints
  power1and2 <- pnorm(c_val)

  # Calculate power for co-primary endpoints
  powerCoprimary <- pbivnorm(x = c_val[1], y = c_val[2], rho = rho)

  # Return results as a data frame
  result <- data.frame(
    n1, n2, p11, p12, p21, p22, rho1, rho2, alpha, Test,
    power1 = power1and2[1], power2 = power1and2[2], powerCoprimary
  )
  class(result) <- c("twoCoprimary", "data.frame")

  return(result)
}
